/*
	PISO-CAN200/400 series device driver.

        Author: Golden Wang

        This program is free software; you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation; either version 2, or (at your option)
        any later version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program; if not, write to the Free Software Foundation,
        Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* File level history (record changes for this file here.)
 *
 *	v 0.0.0 19 Mar 2010 by Golden Wang
 *	Give support to linux kernel 3.x.x.      	
 *
 *      v 0.0.0 19 Mar 2010 by Golden Wang
 *	Linux SocketCAN Driver for PISO-CAN200/400 Series.
 *     
 *     
*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/netdevice.h>
#include <linux/delay.h>
#include <linux/pci.h>
#include <linux/can.h>
#include <linux/can/dev.h>
#include <linux/io.h>
#include <linux/string.h>
#include "sja1000.h"

#define DRV_NAME  "icpdas-ixcan"

MODULE_AUTHOR("Golden <service@icpdas.com>");
MODULE_DESCRIPTION("Socket-CAN driver for ICPDAS PISO-CAN200/400 Series");
MODULE_SUPPORTED_DEVICE("ICPDAS PISO-CAN200/400 cards");
MODULE_LICENSE("GPL v2");

#define IXCAN_PCI_MAX_CHAN 4

struct ixcan_pci_card {
	int version;
	int board;
	int channels;

	struct pci_dev *pci_dev;
	struct net_device *net_dev[IXCAN_PCI_MAX_CHAN];

	void __iomem *conf_addr;
	void __iomem *base_addr;
};

#define IXCAN_PCI_CAN_CLOCK (16000000 / 2)

/*
 * Register definitions for the PLX 9030
 */
#define PLX_ICSR            0x4c   /* Interrupt Control/Status register */
#define PLX_ICSR_LINTI1_ENA 0x0001 /* LINTi1 Enable */
#define PLX_ICSR_PCIINT_ENA 0x0040 /* PCI Interrupt Enable */
#define PLX_ICSR_LINTI1_CLR 0x0400 /* Local Edge Triggerable Interrupt Clear */
#define PLX_ICSR_ENA_CLR    (PLX_ICSR_LINTI1_ENA | PLX_ICSR_PCIINT_ENA )

/*
 * The board configuration is probably following:
 * RX1 is connected to ground.
 * TX1 is not connected.
 * CLKO is not connected.
 * Setting the OCR register to 0xDA is a good idea.
 * This means normal output mode, push-pull and the correct polarity.
 */
#define IXCAN_PCI_OCR         (OCR_TX0_PUSHPULL | OCR_TX1_PUSHPULL)

/*
 * In the CDR register, you should set CBP to 1.
 * You will probably also want to set the clock divider value to 7
 * (meaning direct oscillator output) because the second SJA1000 chip
 * is driven by the first one CLKOUT output.
 */
#define IXCAN_PCI_CDR             (CDR_CBP | CDR_CLKOUT_MASK)

#define IXCAN_VENDOR_ID1        0x10b5

#define IXCAN_DEVICE_ID1        0x9030
#define IXCAN_DEVICE_ID2        0x9050

#define IXCAN_SUBVENDOR_ID1     0x2129

#define IXCAN_SUBDEVICE_PISO_CAN200     0xc200
#define IXCAN_SUBDEVICE_PISO_CAN400     0xc400
#define IXCAN_SUBDEVICE_PISO_CAN200U    0xc210
#define IXCAN_SUBDEVICE_PISO_CAN400U    0xc410
#define IXCAN_SUBDEVICE_PISO_CAN200E    0xc280
#define IXCAN_SUBDEVICE_PISO_CAN400E    0xc480
#define IXCAN_SUBDEVICE_PCM_CAN100      0xc150
#define IXCAN_SUBDEVICE_PCM_CAN200      0xc250
#define IXCAN_SUBDEVICE_PCM_CAN400      0xc450

static unsigned int board_num = 1;
static struct pci_device_id ixcan_pci_tbl[] = {
        /* PISO" means "Pci" interface and "ISOlation"*/
        /* PISO-CAN200, 2-CAN Port*/
        {IXCAN_VENDOR_ID1, IXCAN_DEVICE_ID2, IXCAN_SUBVENDOR_ID1, IXCAN_SUBDEVICE_PISO_CAN200,},

        /* PISO-CAN400, 4-CAN Port */
        {IXCAN_VENDOR_ID1, IXCAN_DEVICE_ID2, IXCAN_SUBVENDOR_ID1, IXCAN_SUBDEVICE_PISO_CAN400,},

        /* PISO-CAN200U, 2-CAN Port, Universal PCI */
        {IXCAN_VENDOR_ID1, IXCAN_DEVICE_ID1, IXCAN_SUBVENDOR_ID1, IXCAN_SUBDEVICE_PISO_CAN200U,},

        /* PISO-CAN400U, 4-CAN Port, Universal PCI */
        {IXCAN_VENDOR_ID1, IXCAN_DEVICE_ID1, IXCAN_SUBVENDOR_ID1, IXCAN_SUBDEVICE_PISO_CAN400U,},

        /* PISO-CAN200E, 2-CAN Port, PCI Express interface*/
        {IXCAN_VENDOR_ID1, IXCAN_DEVICE_ID2, IXCAN_SUBVENDOR_ID1, IXCAN_SUBDEVICE_PISO_CAN200E,},

	/* PCM-CAN100, 1-CAN Port, PC104/PC104+ compliant */
        {IXCAN_VENDOR_ID1, IXCAN_DEVICE_ID1, IXCAN_SUBVENDOR_ID1, IXCAN_SUBDEVICE_PCM_CAN100,},

        /* PCM-CAN200, 2-CAN Port, PC104/PC104+ compliant */
        {IXCAN_VENDOR_ID1, IXCAN_DEVICE_ID1, IXCAN_SUBVENDOR_ID1, IXCAN_SUBDEVICE_PCM_CAN200,},

        /* PCM-CAN400, 4-CAN Port, PC104/PC104+ compliant */
        {IXCAN_VENDOR_ID1, IXCAN_DEVICE_ID1, IXCAN_SUBVENDOR_ID1, IXCAN_SUBDEVICE_PCM_CAN400,},

        { 0,}
};

MODULE_DEVICE_TABLE(pci, ixcan_pci_tbl);

/*
 * Helper to read internal registers from card logic (not CAN)
 */
static u8 ixcan_pci_read_reg(const struct sja1000_priv *priv, int port)
{
        return ioread8(priv->reg_base + port);
}

static void ixcan_pci_write_reg(const struct sja1000_priv *priv,
                                 int port, u8 val)
{
        iowrite8(val, priv->reg_base + port);
	
	if(priv->udelay)
	{
		//printk("Enable UDELAY Flag\n");
                udelay(1);
	}
}

/*
 * Check if a CAN controller is present at the specified location
 * by trying to set 'em into the PeliCAN mode
 */
static inline int ixcan_pci_check_chan(const struct sja1000_priv *priv)
{
	unsigned char res;

	/* Make sure SJA1000 is in reset mode */
	priv->write_reg(priv, REG_MOD, 1);

	priv->write_reg(priv, REG_CDR, CDR_PELICAN);

	/* read reset-values */
	res = priv->read_reg(priv, REG_CDR);

	if (res == CDR_PELICAN)
		return 1;

	return 0;
}

static void ixcan_pci_del_card(struct pci_dev *pdev)
{
	struct ixcan_pci_card *card = pci_get_drvdata(pdev);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,31)
	struct sja1000_priv *priv;
#endif
	struct net_device *dev;
	int i = 0;

	for (i = 0; i < card->channels; i++) 
	{
		dev = card->net_dev[i];
		
		if (!dev)
			continue;
		
		dev_info(&pdev->dev, "Removing %s.\n", dev->name);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,31)
		priv = netdev_priv(dev);

		if(priv->reg_base != NULL)
			pci_iounmap(card->pci_dev, priv->reg_base);

#endif

		unregister_sja1000dev(dev);
		free_sja1000dev(dev);
	}

	if (card->base_addr != NULL)
		pci_iounmap(card->pci_dev, card->base_addr);

	if (card->conf_addr != NULL)
		pci_iounmap(card->pci_dev, card->conf_addr);

	kfree(card);

	pci_disable_device(pdev);
	pci_set_drvdata(pdev, NULL);
}

static void ixcan_pci_card_reset(struct ixcan_pci_card *card)
{
	/* Request board reset */
	writeb(0, card->base_addr);
}

/*
 * Probe PCI device for EMS CAN signature and register each available
 * CAN channel to SJA1000 Socket-CAN subsystem.
 */
static int __devinit ixcan_pci_add_card(struct pci_dev *pdev,const struct pci_device_id *ent)
{
	struct sja1000_priv *priv;
	struct net_device *dev;
	struct ixcan_pci_card *card;
	//int max_chan, conf_size, base_bar;
	int err, i;

	/* Enabling PCI device */
	if (pci_enable_device(pdev) < 0) 
	{
		dev_err(&pdev->dev, "Enabling PCI device failed\n");
		return -ENODEV;
	}

	/* Allocating card structures to hold addresses, ... */
	card = kzalloc(sizeof(struct ixcan_pci_card), GFP_KERNEL);
	if (card == NULL) 
	{
		dev_err(&pdev->dev, "Unable to allocate memory\n");
		pci_disable_device(pdev);
		return -ENOMEM;
	}

	pci_set_drvdata(pdev, card);

	card->pci_dev = pdev;
	card->version = 0;
	card->board = board_num;

	if( ((pdev->subsystem_device >> 8)&0x0f) == 2)
                card->channels = 2;
        else
                card->channels = 4;

	/* Remap configuration space and controller memory area */
	card->conf_addr = pci_iomap(pdev, 0, pci_resource_len(pdev, 0));

	if (card->conf_addr == NULL) 
	{
		err = -ENOMEM;
		goto failure_cleanup;
	}

	card->base_addr = pci_iomap(pdev, 2, pci_resource_len(pdev, 2));

	if (card->base_addr == NULL) 
	{
		err = -ENOMEM;
		goto failure_cleanup;
	}

	ixcan_pci_card_reset(card);

	/* Detect available channels */
	for (i = 0; i < card->channels; i++) 
	{
		dev = alloc_sja1000dev(sizeof(struct ixcan_pci_card));
		if (dev == NULL) 
		{
			err = -ENOMEM;
			goto failure_cleanup;
		}

		card->net_dev[i] = dev;
		priv = netdev_priv(dev);
		priv->priv = card;
		priv->irq_flags = IRQF_SHARED;

		dev->irq = pdev->irq;
		//Bar 0 (PLX), Bar 2 (port 1),  Bar 3 ~ 5   (port 2 ~ 4)
		priv->reg_base = pci_iomap(pdev, (i+2), pci_resource_len(pdev, (i+2)));

		priv->read_reg  = ixcan_pci_read_reg;
		priv->write_reg = ixcan_pci_write_reg;

		priv->vendor = ent->vendor;
		priv->device = ent->device;
		priv->subvendor = ent->subvendor;
		priv->subdevice = ent->subdevice;

		priv->udelay = 0;

		/* Check if channel is present */
		if (ixcan_pci_check_chan(priv)) 
		{
			priv->can.clock.freq = IXCAN_PCI_CAN_CLOCK;
			priv->ocr = IXCAN_PCI_OCR;
			priv->cdr = IXCAN_PCI_CDR;

			SET_NETDEV_DEV(dev, &pdev->dev);

			/* Register SJA1000 device */
			err = register_sja1000dev(dev);
			if (err) 
			{
				dev_err(&pdev->dev, "Registering device failed "
							"(err=%d)\n", err);
				free_sja1000dev(dev);
				goto failure_cleanup;
			}

			dev_info(&pdev->dev, "Board #%d : Channel #%d at 0x%p, irq %d\n", card->board, i + 1, priv->reg_base, dev->irq);
		} 
		else 
		{
			free_sja1000dev(dev);
		}
	}

	iowrite32(PLX_ICSR_ENA_CLR, card->conf_addr + PLX_ICSR);

	board_num++;

	return 0;

failure_cleanup:
	dev_err(&pdev->dev, "Error: %d. Cleaning Up.\n", err);

	ixcan_pci_del_card(pdev);

	return err;
}

static struct pci_driver ixcan_pci_driver = {
	.name = DRV_NAME,
	.id_table = ixcan_pci_tbl,
	.probe = ixcan_pci_add_card,
	.remove = ixcan_pci_del_card,
};

static int __init ixcan_pci_init(void)
{
	return pci_register_driver(&ixcan_pci_driver);
}

static void __exit ixcan_pci_exit(void)
{
	pci_unregister_driver(&ixcan_pci_driver);
}

module_init(ixcan_pci_init);
module_exit(ixcan_pci_exit);

