/* PISO-CAN200-D/T Service Module

   Author: Golden Wang

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

/* File level history (record changes for this file here.)

   v 0.1.0 1 Apr 2011 by Golden Wang
     Give support to linux kernel 2.6.37.

   v 0.0.0 1 May 2009 by Golden Wang
     create, blah blah... */

/* Mandatory */
#include <linux/kernel.h>		/* ta, kernel work */
#include <linux/module.h>		/* is a module */
#include <linux/delay.h>
#include "ixpci.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#include <linux/cdev.h>
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
        /* Deal with CONFIG_MODVERSIONS that is defined in
         * /usr/include/linux/config.h (config.h is included by module.h) */
        #if CONFIG_MODVERSIONS==1
        #define MODVERSIONS
        #include <linux/modversions.h>
        #endif

        /* for compatibility with future version of Linux */
        #include <linux/wrapper.h>
#endif
/* Additional */
#include <linux/fs.h>

/* use I/O ports */
#include <asm/io.h>
#include <linux/ioport.h>

/* irq service */
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
//#include <asm/system.h>
//#include <linux/irqflags.h>

/* Local matter */
#include "_pisocan200.h"

ixpci_devinfo_t *dev;


static int isInitIOMem[DEVICE_NR_DEVS] = {0};
//static int int_filter = 0;
//canmsg_t lastTx_object[4];

#ifdef MODULE_LICENSE
MODULE_AUTHOR("Golden Wang <service@icpdas.com>");
MODULE_DESCRIPTION
	("ICPDAS PCI-series driver, PISO-CAN200-D/T service module");
MODULE_LICENSE(ICPDAS_LICENSE);
#endif

// EXPORT_NO_SYMBOLS; /* do not export symbols */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
EXPORT_SYMBOL_GPL(ixpisocan200_ioctl);
EXPORT_SYMBOL_GPL(ixpisocan200_release);
EXPORT_SYMBOL_GPL(ixpisocan200_open);
EXPORT_SYMBOL_GPL(ixpisocan200_read);
EXPORT_SYMBOL_GPL(ixpisocan200_write);
#else
EXPORT_SYMBOL_NOVERS(ixpisocan200_ioctl);
EXPORT_SYMBOL_NOVERS(ixpisocan200_release);
EXPORT_SYMBOL_NOVERS(ixpisocan200_open);
EXPORT_SYMBOL_NOVERS(ixpisocan200_read);
EXPORT_SYMBOL_NOVERS(ixpisocan200_write);
#endif

/* CAN timing values */
u8 CanTiming[10][2]={
        {CAN_TIM0_10K,  CAN_TIM1_10K},
        {CAN_TIM0_20K,  CAN_TIM1_20K},
        {CAN_TIM0_50K,  CAN_TIM1_50K},
        {CAN_TIM0_100K, CAN_TIM1_100K},
        {CAN_TIM0_125K, CAN_TIM1_125K},
        {CAN_TIM0_250K, CAN_TIM1_250K},
        {CAN_TIM0_500K, CAN_TIM1_500K},
        {CAN_TIM0_800K, CAN_TIM1_800K},
        {CAN_TIM0_1000K,CAN_TIM1_1000K}};

//__inline void _disable_irq(ixpci_devinfo_t * dp)
void _disable_irq(ixpci_devinfo_t * dp)
{
        iowrite8(0, viraddr[dp->cno-1] + 0x4c);
}

//__inline void _enable_irq(ixpci_devinfo_t * dp)
void _enable_irq(ixpci_devinfo_t * dp)
{
        iowrite8(0x00000041, viraddr[dp->cno-1] + 0x4c);
}


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,22)
  irqreturn_t irq_handler(int irq, void *dev_id)
#else
  #if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
  irqreturn_t irq_handler(int irq, void *dev_id, struct pt_regs *regs)
  #else
  void irq_handler(int irq, void *dev_id, struct pt_regs *regs)
  #endif
#endif
{
	int i = 0;
//	int rx_fifo;
//	struct timeval  timestamp;
//	unsigned long flags;
	int ext;                        /* flag for extended message format */
	int status;
	int irqsrc, dummy, bar, port, length, card_no, rtr = 0;
	canfifo_t *RxFifo;
	canfifo_t *TxFifo;
	ixpci_devinfo_t *dp;
	uint8 tx2reg;
        unsigned int id;

	dp = (ixpci_devinfo_t *) dev_id;
        	
	RxFifo = &(dp->caninfo->rx_buf);
	TxFifo = &(dp->caninfo->tx_buf);

	card_no = dp->cno - 1;
	bar = dp->canportid + 1;
        port = dp->canportid;

	if ( (ioread8(dp->ioaddr[bar] + IER) == 0 ))
                return IRQ_NONE;

	irqsrc = ioread8(dp->ioaddr[bar] + IR);

	//Disable Interrupt
	iowrite8(0, viraddr[card_no] + 0x4c);
	(RxFifo->data[RxFifo->head]).flags = (RxFifo->status & BUF_OVERRUN ? MSG_BOVR : 0);

	do{
		//printk("CAN Port %d : irqsrc : 0x%x \n", port, irqsrc);

		//Handle receive interrupt

		if( irqsrc & CAN_RECEIVE_INT )
		{
			//printk("CAN Port %d :  Handle CAN_RECEIVE_INT\n",port);

			dummy = ioread8(dp->ioaddr[bar] + RFINF);

			if( dummy & CAN_RTR )
			{
                		(RxFifo->data[RxFifo->head]).flags |= MSG_RTR;
                                rtr = 1;
                        }
                        else
                        {
                                rtr = 0;
                        }


		        if( dummy & CAN_EFF )
			{
               			(RxFifo->data[RxFifo->head]).flags |= MSG_EXT;
       			}

		        ext = (dummy & CAN_EFF);
			
			if(ext)
			{
				(RxFifo->data[RxFifo->head]).id =
				(((unsigned int)(ioread8(dp->ioaddr[bar] + RFID1))) << 21)
				+(((unsigned int)(ioread8(dp->ioaddr[bar] + RFID2))) << 13)
				+(((unsigned int)(ioread8(dp->ioaddr[bar] + RFID3))) << 5)
				+(((unsigned int)(ioread8(dp->ioaddr[bar] + RFID4))) >> 3);
       			}
			else
			{
                		(RxFifo->data[RxFifo->head]).id =
				(((unsigned int)(ioread8(dp->ioaddr[bar] + RFID1))) << 3)
                               	+(((unsigned int)(ioread8(dp->ioaddr[bar] + RFID2))) >> 5);
	        	}

			length = dummy  & 0x0f;                    
     			(RxFifo->data[RxFifo->head]).length = length;

			length %= 9;        

			memset((RxFifo->data[RxFifo->head]).data, 0,sizeof((RxFifo->data[RxFifo->head]).data));

                        if (!rtr )
			{
			        for( i = 0; i < length; i++)
				{
			        
       	        			if(ext)
					{		
	                			(RxFifo->data[RxFifo->head]).data[i] = ioread8(dp->ioaddr[bar] +(RFEFDB+i));
	                		}
					else
					{
		                		(RxFifo->data[RxFifo->head]).data[i] = ioread8(dp->ioaddr[bar] +(RFSFDB+i));
               				}
				}
			}

		        RxFifo->status = BUF_OK;
		        RxFifo->head = ++(RxFifo->head) % MAX_BUFSIZE;
		        if(RxFifo->head == RxFifo->tail)
			{
               			//printk("CAN[%d][%d] RX: FIFO overrun\n", minor, rx_fifo);
		                RxFifo->status = BUF_OVERRUN;
		        }
			wake_up_interruptible(&CanWait[card_no][port-1]);

			iowrite8(CAN_RELEASE_RECEIVE_BUFFER, dp->ioaddr[bar] + CMR);

        		if((ioread8(dp->ioaddr[bar] + SR) & CAN_DATA_OVERRUN))
			{
               	 		//printk("CAN Port %d Rx: Overrun Status \n", port);
				iowrite8(CAN_CLEAR_OVERRUN_STATUS, dp->ioaddr[bar] + CMR);
			} 

		}

		//Handle transmit interrupt 

		if( irqsrc & CAN_TRANSMIT_INT )
		{
  			 //printk("CAN Port %d :  Handle CAN_TRANSMIT_INT \n",port);

			 if (CanWaitFlag[card_no][port-1] == 1)
			 {

/*
		                memcpy(
			                    (void *)&RxFifo->data[RxFifo->head],
			                    (void *)&lastTx_object[dp->no],
			                    sizeof(canmsg_t));

                		RxFifo->data[RxFifo->head].flags |= MSG_SELF;

		                RxFifo->status = BUF_OK;
		                RxFifo->head = ++(RxFifo->head) % MAX_BUFSIZE;
				//printk("irq_handler TXINT RxFifo->head : %d\n",RxFifo->head);

                		if(RxFifo->head == RxFifo->tail) {
		                    //printk("CAN Port %d RX: FIFO overrun\n", port);
                		    RxFifo->status = BUF_OVERRUN;
                		}
*/
                		wake_up_interruptible(&CanWait[card_no][port-1]);
								
			 }


			 if( TxFifo->free[TxFifo->tail] == BUF_EMPTY )
			 {
		         	TxFifo->status = BUF_EMPTY;
		         	TxFifo->active = 0;
			        wake_up_interruptible(&CanOutWait[card_no][port-1]);
				goto Tx_done;
			 }
/*
			 memcpy(
		                (void *)&lastTx_object[dp->no],
                		(void *)&TxFifo->data[TxFifo->tail],
		                sizeof(canmsg_t));
*/
			tx2reg = (TxFifo->data[TxFifo->tail]).length;
		        if( (TxFifo->data[TxFifo->tail]).flags & MSG_RTR) 
			{
                		tx2reg |= CAN_RTR;
				rtr = 1;
                        }
                        else
                        {
                                rtr = 0;
                        }

		        ext = (TxFifo->data[TxFifo->tail]).flags & MSG_EXT;
		        id = (TxFifo->data[TxFifo->tail]).id;

			if(ext)
	                {
                        	iowrite8(CAN_EFF + tx2reg, dp->ioaddr[bar] + TFINF);
                        	iowrite8((uint8)(id >> 21), dp->ioaddr[bar] + TFID1);
                        	iowrite8((uint8)(id >> 13), dp->ioaddr[bar] + TFID2);
                        	iowrite8((uint8)(id >> 5), dp->ioaddr[bar] + TFID3);
				if (rtr)
                                        iowrite8(((uint8)(id << 3) & 0xff) | 0x04, dp->ioaddr[bar] + TFID4);
                                else
                                        iowrite8((uint8)(id << 3) & 0xff, dp->ioaddr[bar] + TFID4);

                	}
                	else
                	{
                        	iowrite8(CAN_SFF + tx2reg, dp->ioaddr[bar] + TFINF);
                        	iowrite8((uint8)(id >> 3), dp->ioaddr[bar] + TFID1);
				if (rtr)
                                        iowrite8((((uint8)(id << 5) & 0xff) | 0x10), dp->ioaddr[bar] + TFID2);
                                else
                                        iowrite8(((uint8)(id << 5) & 0xe0), dp->ioaddr[bar] + TFID2);

                	}

			tx2reg &= 0x0f;         

	                if(ext)
        	        {
                	        for( i=0; i < tx2reg; i++)
                        	{
                                	iowrite8((TxFifo->data[TxFifo->tail]).data[i], dp->ioaddr[bar] +(TFEFDB+i));
					udelay(5);
                        	}
                	}
                	else
                	{
                        	for( i=0; i < tx2reg ; i++)
                        	{
 		                      	//printk("(TxFifo->data[TxFifo->tail]).data[%d] : 0x%x\n",i ,(TxFifo->data[TxFifo->tail]).data[i]);
	                                iowrite8((TxFifo->data[TxFifo->tail]).data[i], dp->ioaddr[bar] +(TFSFDB+i));
					udelay(5);
                        	}
                	}

        	        iowrite8(CAN_TRANSMISSION_REQUEST, dp->ioaddr[bar] + CMR);

		        TxFifo->free[TxFifo->tail] = BUF_EMPTY; 

		        TxFifo->tail = ++(TxFifo->tail) % MAX_BUFSIZE;
		}
Tx_done:
		if( irqsrc & CAN_ERROR_INT )
		{

		}

		if( irqsrc & CAN_OVERRUN_INT )
		{
			printk("CAN Port %d : controller overrun!\n", port);
                        //Overrun[minor]++;

                        /* insert error */
                        status = ioread8(dp->ioaddr[bar] + SR);

                        if (CanWaitFlag[card_no][port-1] == 1)
                        {
                                /* this FIFO is in use */
                                if(status & CAN_DATA_OVERRUN)
                                {
                                        (RxFifo->data[RxFifo->head]).flags += MSG_OVR;
                                }
                                (RxFifo->data[RxFifo->head]).id = 0xFFFFFFFF;
                                (RxFifo->data[RxFifo->head]).length = 0;
                                RxFifo->status = BUF_OK;
                                RxFifo->head = ++(RxFifo->head) % MAX_BUFSIZE;
                                if(RxFifo->head == RxFifo->tail)
                                {
                                        printk("CAN Port %d RX: FIFO overrun\n", port);
                                        RxFifo->status = BUF_OVERRUN;
                                }
                                /* tell someone that there is a new error message */
                                wake_up_interruptible(&CanWait[card_no][port-1]);
                        }

                        iowrite8(CAN_CLEAR_OVERRUN_STATUS, dp->ioaddr[bar] + CMR);
		}
		

	} while( (irqsrc = ioread8(dp->ioaddr[bar] + IR)) != 0);

	//Enable interrupt
	iowrite8(0x00000041, viraddr[card_no] + 0x4c);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
        return IRQ_HANDLED;
#endif

}

int _write_reg(ixpci_canreg_t * reg, void __iomem *base[], unsigned int bar)
//int _write_reg(ixpci_reg_t * reg, unsigned long int base[])
{
	/* Write to register
	 *
	 * Arguments:
	 *   reg      pointer to a structure ixpci_reg for register
	 *   base     base address for current device
	 *
	 * Returned:  SUCCESS or FAILURE */
	int i, ext;
	uint8 tx2reg;
	int rtr = 0;
	unsigned int timeout = 0;
	unsigned int result = SUCCESS;
	unsigned int status;

	switch (reg->id) {

	case IXPCI_CAN_CONFIG:

		switch(reg->baud)
		{
                	case   10: i = 0; break;
			case   20: i = 1; break;
			case   50: i = 2; break;
			case  100: i = 3; break;
			case  125: i = 4; break; 
			case  250: i = 5; break; // default CAN baud
			case  500: i = 6; break;
			case  800: i = 7; break;
			case 1000: i = 8; break;
			default  : i = 5; break;
		}

		//printk("Port : %d --> Baud = %d ACR = 0x%x AMR = 0x%x\n", (bar-1), reg->baud,  reg->acr, reg->amr);
		//set CR.0=1 reset mode
		iowrite8(CAN_RESET_REQUEST + CAN_MODE_DEF, base[bar] + MOD);
		timeout = 0;
        	while(++timeout < CAN_TIMEOUT_CNT)
        	{
                	if( ioread8(base[bar] + MOD) & CAN_RESET_REQUEST ) break;
        	}

		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		//set ACR, AMR
	        iowrite32(reg->acr, base[bar] + PACR);
		timeout = 0;
        	while(++timeout < CAN_TIMEOUT_CNT)
        	{
                	if( ioread32(base[bar] + PACR) == reg->acr  ) break;
        	}
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

	        iowrite32(reg->amr, base[bar] + PAMR);
		timeout = 0;
        	while(++timeout < CAN_TIMEOUT_CNT)
        	{
                	if( ioread32(base[bar] + PAMR) == reg->amr  ) break;
        	}
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;
		
		//set BTR0, BTR1
		iowrite8(CanTiming[i][0], base[bar] + BTR0);
		timeout = 0;
        	while(++timeout < CAN_TIMEOUT_CNT)
        	{
                	if( ioread8(base[bar] + BTR0 ) == CanTiming[i][0]  ) break;
        	}
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		iowrite8(CanTiming[i][1], base[bar] + BTR1);
		timeout = 0;
        	while(++timeout < CAN_TIMEOUT_CNT)
        	{
                	if( ioread8(base[bar] + BTR1 ) == CanTiming[i][1]  ) break;
        	}
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		//set OCR
		iowrite8(CAN_OUTC_VAL, base[bar] + OCR);
		timeout = 0;
        	while(++timeout < CAN_TIMEOUT_CNT)
        	{
                	if( ioread8(base[bar] + OCR ) == CAN_OUTC_VAL ) break;
        	}
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		//set RBSA
		iowrite8(0, base[bar] + RBSA);
		timeout = 0;
        	while(++timeout < CAN_TIMEOUT_CNT)
        	{
                	if( ioread8(base[bar] + RBSA) == 0x00 ) break;
        	}
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		// return to operating mode */
	        iowrite8( (ioread8(base[bar]+MOD) & ~CAN_RESET_REQUEST), base[bar] + MOD );
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( !(ioread8(base[bar] + MOD) & CAN_RESET_REQUEST) ) break;
                }
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

                break;

	case IXPCI_CAN_STARTCHIP:
		//set CR.0=1 reset mode
                iowrite8(CAN_RESET_REQUEST + CAN_MODE_DEF, base[bar] + MOD);

		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( ioread8(base[bar] + MOD) & CAN_RESET_REQUEST ) break;
                }

                if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		/* clear interrupts */
	        ioread8(base[bar] + IR);

        	/* Interrupts on Rx, TX, any Status change and data overrun */

        	//iowrite8( (CAN_OVERRUN_INT_ENABLE | CAN_ERROR_INT_ENABLE | CAN_TRANSMIT_INT_ENABLE | CAN_RECEIVE_INT_ENABLE), base[bar] + IER);		

		 // return to operating mode */

                iowrite8( (ioread8(base[bar] + MOD) & ~CAN_RESET_REQUEST), base[bar]+MOD );
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( !(ioread8(base[bar] + MOD) & CAN_RESET_REQUEST) ) break;
                }
                if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		break;

	case IXPCI_CAN_STOPCHIP:
		//set CR.0=1 reset mode
                iowrite8(CAN_RESET_REQUEST +CAN_MODE_DEF, base[bar] + MOD);
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( ioread8(base[bar] + MOD) & CAN_RESET_REQUEST ) break;
                }
                if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		break;

	case IXPCI_CAN_INITCHIP:
                //set CR.0=1 reset mode
                iowrite8(CAN_RESET_REQUEST + CAN_MODE_DEF, base[bar] + MOD);
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( ioread8(base[bar] + MOD) & CAN_RESET_REQUEST ) break;
                }
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		// IER : disable CAN interrupt
                iowrite8(0x00, base[bar] + IER );

                timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( ioread8(base[bar] + IER) == 0x00 ) break;
                }
                if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;
		//status = ioread8(base[bar] + SR);

	        /* choose BasicCAN or PeliCAN mode */
        	iowrite8(CAN_MODE_PELICAN + CAN_MODE_CLK, base[bar] + CDR);

	        timeout = 0;
        	while(++timeout < CAN_TIMEOUT_CNT)
        	{
                	if( ioread8(base[bar] + CDR) == (CAN_MODE_PELICAN + CAN_MODE_CLK) ) break;
        	}
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

	        iowrite8(CAN_OUTC_VAL, base[bar] + OCR);
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( ioread8(base[bar] + OCR ) == CAN_OUTC_VAL ) break;
                }
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		/* set CAN sja1000 default baud 250K */
		iowrite8(CAN_TIM0_250K, base[bar] + BTR0);
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( ioread8(base[bar] + BTR0 ) == CAN_TIM0_250K  ) break;
                }
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

	        iowrite8(CAN_TIM1_250K, base[bar] + BTR1);
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( ioread8(base[bar] + BTR1 ) == CAN_TIM1_250K  ) break;
                }
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

        	/*set default ACR, AMR */
	        iowrite32(0, base[bar] + PACR);
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( ioread32(base[bar] + PACR) == 0  ) break;
                }
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

        	iowrite32(STD_MASK, base[bar] + PAMR);
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( ioread32(base[bar] + PAMR) == STD_MASK  ) break;
                }
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		// Interrupt on TX and data overrun
                iowrite8( (ioread8(base[bar] + IER) | CAN_TRANSMIT_INT_ENABLE | CAN_OVERRUN_INT_ENABLE), base[bar] + IER);
                timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if(ioread8(base[bar] + IER) == (CAN_TRANSMIT_INT_ENABLE | CAN_OVERRUN_INT_ENABLE))  break;
                }
                if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

		// return to operating mode */
                iowrite8( (ioread8(base[bar] + MOD) & ~CAN_RESET_REQUEST), base[bar]+MOD );
		timeout = 0;
                while(++timeout < CAN_TIMEOUT_CNT)
                {
                        if( !(ioread8(base[bar] + MOD) & CAN_RESET_REQUEST) ) break;
                }
		if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

                break;

	case IXPCI_CAN_ENABLETXIRQ:
		if ( !(ioread8(base[bar] + IER) & CAN_TRANSMIT_INT_ENABLE) )
		{
			iowrite8( (ioread8(base[bar] + IER) | CAN_TRANSMIT_INT_ENABLE), base[bar] + IER);
		        timeout = 0;
        		while(++timeout < CAN_TIMEOUT_CNT)
        		{
                		if(ioread8(base[bar] + IER) & CAN_TRANSMIT_INT_ENABLE)  break;
        		}
			if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;
		}
		break;

	case IXPCI_CAN_DISABLETXIRQ:
		if ( (ioread8(base[bar] + IER) & CAN_TRANSMIT_INT_ENABLE) )
                {
                        iowrite8( (ioread8(base[bar] + IER) & 0xfd), base[bar] + IER);

                        timeout = 0;
                        while(++timeout < CAN_TIMEOUT_CNT)
                        {
                                if( !(ioread8(base[bar] + IER) & CAN_TRANSMIT_INT_ENABLE) ) break;
                        }
                        if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;
                }

		break;

	case IXPCI_CAN_ENABLERXIRQ:
		if ( !(ioread8(base[bar] + IER) & CAN_RECEIVE_INT_ENABLE) )
		{
			iowrite8( (ioread8(base[bar] + IER) | CAN_RECEIVE_INT_ENABLE), base[bar] + IER);	
			timeout = 0;
        		while(++timeout < CAN_TIMEOUT_CNT)
        		{
                		if( (ioread8(base[bar] + IER) & CAN_RECEIVE_INT_ENABLE) ) break;
        		}
			if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;
		}
		
		break;

	case IXPCI_CAN_DISABLERXIRQ:
                if ( (ioread8(base[bar] + IER) & CAN_RECEIVE_INT_ENABLE) )
                {
                        iowrite8( (ioread8(base[bar] + IER) & 0xfe), base[bar] + IER);

			timeout = 0;
                        while(++timeout < CAN_TIMEOUT_CNT)
                        {
                                if( !(ioread8(base[bar] + IER) & CAN_RECEIVE_INT_ENABLE) ) break;
                        }
                        if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;
                }

                break;
	
	case IXPCI_CAN_CLEARBUFFERS:
		break;

	case IXPCI_CAN_CLEAROVERRUN:

		status = ioread8(base[bar] + SR);	
		
		if ( !(status & CAN_DATA_OVERRUN))
		{
			result = SUCCESS;
			break;
		}
		else
		{
			iowrite8(CAN_CLEAR_OVERRUN_STATUS, base[bar] + CMR);
			timeout = 0;
			do
			{
                		status = ioread8(base[bar] + SR);  // Get CAN Status
                        
				if(!(status & CAN_DATA_OVERRUN))
					break;
		        }while(++timeout < CAN_TIMEOUT_CNT);

			if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;
		}

		break;

	case IXPCI_CAN_SENDMSG:
		reg->tx.length %= 9;              
		ext = (reg->tx.flags & MSG_EXT);   
		
		tx2reg = reg->tx.length;
		if( reg->tx.flags & MSG_RTR) 
		{
			tx2reg |= CAN_RTR;
			rtr = 1;
		}

		if ( reg->tx.id == 0 ) {
			printk(KERN_INFO "SEND 0x%lx %x %x\n", reg->tx.id, reg->tx.data[0],
			reg->tx.data[1]);
		}	

		if(ext)
		{
			//printk("Data frame : Extended frame format\n");
			iowrite8(CAN_EFF + tx2reg, base[bar] + TFINF);
			iowrite8((uint8)(reg->tx.id >> 21), base[bar] + TFID1);
			iowrite8((uint8)(reg->tx.id >> 13), base[bar] + TFID2);
			iowrite8((uint8)(reg->tx.id >> 5), base[bar] + TFID3);
			if (rtr)
                               iowrite8((((uint8)(reg->tx.id << 3) & 0xff) | 0x04), base[bar] + TFID4);
                        else
                               iowrite8((uint8)(reg->tx.id << 3) & 0xff, base[bar] + TFID4);
		}
		else
		{
			//printk("Data frame : Standard frame format\n");
			iowrite8(CAN_SFF + tx2reg, base[bar] + TFINF);
                        iowrite8((uint8)(reg->tx.id >> 3), base[bar] + TFID1);
			if (rtr)
                                iowrite8((((uint8)(reg->tx.id << 5) & 0xff) | 0x10), base[bar] + TFID2);
                        else
                                iowrite8(((uint8)(reg->tx.id << 5) & 0xe0), base[bar] + TFID2);
		}

		if(ext)
		{
			for( i=0; i < reg->tx.length ; i++)
			{
				iowrite8(reg->tx.data[i], base[bar] +(TFEFDB+i));
				udelay(5);
        		}
   		}
		else
		{
        		for( i=0; i < reg->tx.length ; i++)
			{
				//printk("reg->tx.data[%d] : 0x%x\n",i ,reg->tx.data[i]);
				iowrite8(reg->tx.data[i], base[bar] +(TFSFDB+i));
				udelay(5);
			}
		}
	
		iowrite8(CAN_TRANSMISSION_REQUEST, base[bar] + CMR);

		break;
	
	default:
		return FAILURE;
	}

	return result;
}

//int _read_reg(ixpci_reg_t * reg, void __iomem * base[], unsigned int bar, unsigned int baud)
int _read_reg(ixpci_canreg_t * reg, ixpci_devinfo_t * dp)
{
	/* Read from register
	 *
	 * Arguments:
	 *   reg      pointer to structure ixpci_reg for register
	 *   base     base address for current device
	 *
	 * Returned:  SUCCESS or FAILURE */

	int ext, length, i;
       	uint8 rfinfo = 0, status;
	canfifo_t *RxFifo;
	canfifo_t *TxFifo;
	//unsigned long flags;
	int rtr = 0;
	unsigned int bar = dp->canportid+1;
//	void * base = dp->ioaddr[bar];
	unsigned int baud = dp->caninfo->baud;
	unsigned int card_no = dp->cno -1;
	
	RxFifo = &(dp->caninfo->rx_buf);
        TxFifo = &(dp->caninfo->tx_buf);

	switch (reg->id) {

	case IXPCI_CAN_STATUS:
		reg->stat.type = CAN_TYPE_SJA1000;
		reg->stat.baud = baud;
		/* printk(" STAT ST %d\n", CANin(board, canstat)); */
		reg->stat.status = ioread8(dp->ioaddr[bar] + SR) & _SR_MASK;
		/* printk(" STAT EWL %d\n", CANin(board, errorwarninglimit)); */
		reg->stat.error_warning_limit = ioread8(dp->ioaddr[bar] + EWLR);
		reg->stat.rx_errors = ioread8(dp->ioaddr[bar] + RXERR);
		reg->stat.tx_errors = ioread8(dp->ioaddr[bar] + TXERR);
		reg->stat.error_code= ioread8(dp->ioaddr[bar] + ECC);
		//local_irq_save(flags);
		//Disable Interrupt
	        iowrite8(0, viraddr[card_no] + 0x4c);

		//printk("RxFifo->head : %d <--> RxFifo->tail : %d\n",RxFifo->head, RxFifo->tail);
		reg->stat.rx_buffer_size = MAX_BUFSIZE; /**< size of rx buffer  */
		/* number of messages */
		reg->stat.rx_buffer_used = (RxFifo->head < RxFifo->tail)? (MAX_BUFSIZE - RxFifo->tail + RxFifo->head) : (RxFifo->head - RxFifo->tail);
    		reg->stat.tx_buffer_size = MAX_BUFSIZE; /* size of tx buffer  */
		/* number of messages */
		//printk("TxFifo->head : %d <--> TxFifo->tail : %d\n",TxFifo->head, TxFifo->tail);
		reg->stat.tx_buffer_used = (TxFifo->head < TxFifo->tail)? (MAX_BUFSIZE - TxFifo->tail + TxFifo->head) : (TxFifo->head - TxFifo->tail);
		/* Enable CAN Interrupts */
		/* !!!!!!!!!!!!!!!!!!!!! */
		/* restore_flags(flags); */
		//local_irq_restore(flags);		
		//Enable interrupt
	        iowrite8(0x00000041, viraddr[card_no] + 0x4c);

		break;
	case IXPCI_CAN_IRQSTATUS:
                reg->value = ioread8(dp->ioaddr[bar] + IER);
                break;
	case IXPCI_CAN_RXIRQSTATUS:
		reg->value = ioread8(dp->ioaddr[bar] + IER) & CAN_RECEIVE_INT_ENABLE;
		break;
	case IXPCI_CAN_RXMSGCOUNT:
                if ( ioread8(dp->ioaddr[bar] + IER) & CAN_RECEIVE_INT_ENABLE )
                {
                        if ( RxFifo->head == RxFifo->tail)
                                reg->value = 0;
			else if ( RxFifo->head < RxFifo->tail )
                                reg->value = MAX_BUFSIZE - RxFifo->tail + RxFifo->head;
                        else
                                reg->value = RxFifo->head - RxFifo->tail;
                }
                else
                {
                        reg->value = ioread8(dp->ioaddr[bar] + RMC);
                }
                break;
	case IXPCI_CAN_RECEMSG:

                status = ioread8(dp->ioaddr[bar] + SR);

		//printk("ioctl command IXPCI_CAN_RECEMSG SR : 0x%x\n",status);
                if( status & CAN_RECEIVE_BUFFER_STATUS )
                {
                        rfinfo= ioread8(dp->ioaddr[bar] + RFINF);
			length = rfinfo & 0x0f;
                        reg->rx.length = length;

			if ( reg->rx.flags ) reg->rx.flags = 0;

			if( rfinfo & CAN_RTR )
			{
                                reg->rx.flags |= MSG_RTR;
				rtr = 1;
			}

                        if( rfinfo & CAN_EFF )
                                reg->rx.flags |= MSG_EXT;

			reg->rx.flags = reg->rx.flags & 0xcf;

                        ext = (rfinfo & CAN_EFF);

			if(ext)
                        {
                                reg->rx.id =
                                (((unsigned int)(ioread8(dp->ioaddr[bar] + RFID1))) << 21)
                                + (((unsigned int)(ioread8(dp->ioaddr[bar]+ RFID2))) << 13)
                                + (((unsigned int)(ioread8(dp->ioaddr[bar]+ RFID3))) << 5)
                                + (((unsigned int)(ioread8(dp->ioaddr[bar]+ RFID4))) >> 3);
                        }
                        else
                        {
                                reg->rx.id =
                                (((unsigned int)(ioread8(dp->ioaddr[bar] + RFID1))) << 3)
                                + (((unsigned int)(ioread8(dp->ioaddr[bar]+ RFID2))) >> 5);
                        }

			length %= 9;

			if (!rtr)
                        {
				for( i = 0; i < length; i++)
	                        {
        	                        if(ext)
                	                {
                        	                reg->rx.data[i] = ioread8(dp->ioaddr[bar] +(RFEFDB+i));
						//printk("rx.data[%d] : 0x%x\n",i,reg->rx.data[i]);
                                	}
                                	else
                                	{
                                        	reg->rx.data[i] = ioread8(dp->ioaddr[bar] +(RFSFDB+i));
						//printk("rx.data[%d] : 0x%x\n",i,reg->rx.data[i]);
        	                        }
                	        }
			}
			else
			{
				reg->rx.data[0] = 0;
	                        reg->rx.data[1] = 0;
        	                reg->rx.data[2] = 0;
                	        reg->rx.data[3] = 0;
                        	reg->rx.data[4] = 0;
	                        reg->rx.data[5] = 0;
        	                reg->rx.data[6] = 0;
                	        reg->rx.data[7] = 0;

			}

			iowrite8((CAN_RELEASE_RECEIVE_BUFFER), dp->ioaddr[bar] + CMR);
			//iowrite8((CAN_RELEASE_RECEIVE_BUFFER |  CAN_CLEAR_OVERRUN_STATUS), dp->ioaddr[bar] + CMR);
                }
		else
		{
			//no data in receive buffer
			reg->rx.flags = 0;
		        reg->rx.length = 0;
		        reg->rx.id=0;
			reg->rx.data[0] = 0;
		        reg->rx.data[1] = 0;
		        reg->rx.data[2] = 0;
		        reg->rx.data[3] = 0;
		        reg->rx.data[4] = 0;
		        reg->rx.data[5] = 0;
		        reg->rx.data[6] = 0;
		        reg->rx.data[7] = 0;
			return FAILURE;
		}

                break;
	default:
		return FAILURE;

	}
	return SUCCESS;
}

ixpci_devinfo_t *_align_minor(int minor)
{
	/* align to device by minor number */

	ixpci_devinfo_t *dp;
	for (dp = dev; dp && dp->no != minor; dp = dp->next_f);
	return dp;
}

ssize_t ixpisocan200_read(struct file *file, char *buffer, size_t count, loff_t *offset)
{
	ixpci_devinfo_t *dp = file->private_data;
	canfifo_t *RxFifo;
	canmsg_t *addr;
	int blocking;
	int written = 0;
	unsigned int port = dp->canportid;
	unsigned int card_no = dp->cno - 1;
	
//	unsigned int bar = dp->canportid+1;

	RxFifo = &(dp->caninfo->rx_buf);
	blocking = !(file->f_flags & O_NONBLOCK);
        addr = (canmsg_t *)buffer;


	while( written < count )
	{

		//printk("ixpisocan200_read RxFifo->tail : %d RxFifo->head : %d\n",RxFifo->tail, RxFifo->head);
            	if( RxFifo->tail == RxFifo->head )
		{
                	RxFifo->status = BUF_EMPTY;

                	if(blocking)
			{
				if(wait_event_interruptible(CanWait[card_no][port-1], RxFifo->tail != RxFifo->head))
                        	{
                                	return -ERESTARTSYS;
                        	}
                	}
			else
			{
                    		break;
			}

		}

		if(copy_to_user( (canmsg_t *) &(addr[written]), (canmsg_t *) &(RxFifo->data[RxFifo->tail]), sizeof(canmsg_t) ))
		{
			written = -EFAULT;
                        goto can_read_exit;
			
		}
            	written++;
		RxFifo->tail = ++(RxFifo->tail) % MAX_BUFSIZE;
        }
can_read_exit:    

	return(written);
}

ssize_t ixpisocan200_write(struct file *file, const char *buffer, size_t count, loff_t *offset)
{
	ixpci_devinfo_t *dp = file->private_data; 
	canfifo_t *TxFifo;
	canmsg_t *addr;
	canmsg_t tx;
	int written        = 0;
	int blocking;
	int i,ext;                        /* message format to send */
	uint8 tx2reg;
	int rtr = 0;
	unsigned int bar = dp->canportid+1;
	unsigned int port = dp->canportid;
	unsigned int card_no = dp->cno - 1;

        TxFifo = &(dp->caninfo->tx_buf);

	spin_lock(&write_splock[card_no][port-1]);
	/* detect write mode */
	blocking = !(file->f_flags & O_NONBLOCK);

	addr = (canmsg_t *)buffer;

	//Disable interrupt
        iowrite8(0, viraddr[card_no] + 0x4c);

	while( written < count )
	{
	        if(blocking)
		{

            		if(wait_event_interruptible(CanOutWait[card_no][port-1],TxFifo->free[TxFifo->head] != BUF_FULL))
			{
	                	written = -ERESTARTSYS;
        	        	goto can_write_exit;
			}
	        }
		else
		{
            		/* there are data to write to the network */
		        if(TxFifo->free[TxFifo->head] == BUF_FULL)
			{
		                /* there is already one message at this place */;
		                /* local_irq_restore(flags); */
		                /* DBGout(); */
		                /* return -ENOSPC; */
		                /* return written; */
		                goto can_write_exit;
            		}
        	}

		if( TxFifo->active )
		{
		        /* more than one data and actual data in queue,
   	                 * add this message to the Tx queue
            		 */
		        if( copy_from_user((canmsg_t *) &(TxFifo->data[TxFifo->head]),
			                   (canmsg_t *) &addr[written],
		           	           sizeof(canmsg_t) ) )
			{
				written = -EFAULT;
				goto can_write_exit;
			}

	                TxFifo->free[TxFifo->head] = BUF_FULL; /* now this entry is FULL */
            		TxFifo->head = ++(TxFifo->head) % MAX_BUFSIZE;
			//printk("ixpisocan200_write TxFifo->head : %d\n",TxFifo->head);
        	}
		else
		{
            		/* copy message into local acnmsg_t structure */
			if( copy_from_user((canmsg_t *) &tx,
		   	                   (canmsg_t *) &addr[written],
		                           sizeof(canmsg_t) ) )
			{
				written = -EFAULT;
                                goto can_write_exit;
			}
            		/* f - fast -- use interrupts */
		        if( count >= 1 )
			{
                		/* !!! CHIP abh. !!! */
				//printk("change TxFifo->active =1 \n");
		                TxFifo->active = 1;
        		}
        	        /* write CAN msg data to the chip and enable the tx interrupt */

			tx.length %= 9;               /* limit CAN message length to 8 */
                	ext = (tx.flags & MSG_EXT);   /* read message format */

	                /* fill the frame info and identifier fields */
        	        tx2reg = tx.length;
	                if( tx.flags & MSG_RTR)
			{
                        	tx2reg |= CAN_RTR;
				rtr = 1;
                        }
                        else
                        {
                                rtr = 0;
                        }

	                if ( tx.id == 0 )
			{
	                        printk(KERN_INFO "SEND 0x%lx %x %x\n", tx.id, tx.data[0],
        	                tx.data[1]);
	                }

			if(ext)
                	{
                        	iowrite8(CAN_EFF + tx2reg, dp->ioaddr[bar] + TFINF);
	                        iowrite8((uint8)(tx.id >> 21), dp->ioaddr[bar] + TFID1);
        	                iowrite8((uint8)(tx.id >> 13), dp->ioaddr[bar] + TFID2);
                	        iowrite8((uint8)(tx.id >> 5), dp->ioaddr[bar] + TFID3);
				if (rtr)
                                        iowrite8( (((uint8)(tx.id << 3) & 0xff) | 0x04), dp->ioaddr[bar] + TFID4);
                                else
                                        iowrite8((uint8)(tx.id << 3) & 0xff, dp->ioaddr[bar] + TFID4);
                	}
                	else
                	{
	                        iowrite8(CAN_SFF + tx2reg, dp->ioaddr[bar] + TFINF);
        	                iowrite8((uint8)(tx.id >> 3), dp->ioaddr[bar] + TFID1);
				if (rtr)
                                        iowrite8( (((uint8)(tx.id << 5) & 0xff) | 0x10), dp->ioaddr[bar] + TFID2);
                                else
                                        iowrite8(((uint8)(tx.id << 5) & 0xe0), dp->ioaddr[bar] + TFID2);

                	}
			
			/* - fill data ---------------------------------------------------- */
	                if(ext)
        	        {
                	        for( i=0; i < tx.length ; i++)
                        	{
                                	iowrite8(tx.data[i], dp->ioaddr[bar] +(TFEFDB+i));
					udelay(5);
                        	}
                	}
                	else
                	{
                        	for( i=0; i < tx.length ; i++)
                        	{
					 //printk("ixpisocan200_write tx.data[%d] : %x\n",i,tx.data[i]);
                                	iowrite8(tx.data[i], dp->ioaddr[bar] +(TFSFDB+i));
					udelay(5); 
                        	}
                	}

	                /* - /end ------------------------------------------------------ */
	                iowrite8(CAN_TRANSMISSION_REQUEST, dp->ioaddr[bar] + CMR);

//			memcpy((void *)&lastTx_object[dp->no],(void *)&tx,sizeof(canmsg_t));

        	}       /* TxFifo->active */
	
	        written++;
	}

can_write_exit:

	//Enable interrupt
        iowrite8(0x00000041, viraddr[card_no] + 0x4c);

	spin_unlock(&write_splock[card_no][port-1]);
		
	return written;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
long ixpisocan200_ioctl(struct file *file, unsigned int ioctl_num, unsigned long ioctl_param)
#else
int ixpisocan200_ioctl(struct inode *inode, struct file *file, unsigned int ioctl_num, unsigned long ioctl_param)
#endif
{
	/* (export)
	 *
	 * This function is called by ixpci.o whenever a process tries
	 * to do and IO control on IXPCI device file
	 *
	 * Arguments: read <linux/fs.h> for (*ioctl) of struct file_operations
	 *
	 * Returned:  SUCCESS or FAILED */

	ixpci_devinfo_t *dp;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
        dp = file->private_data;
#else
        dp = _align_minor(MINOR(inode->i_rdev));
#endif

	if (!dp || !dp->open) return -EINVAL;

	switch (ioctl_num) {
	case IXPCI_GET_INFO:
		ixpci_copy_devinfo((ixpci_devinfo_t *) ioctl_param, dp);
		break;
	case IXPCI_READ_REG:
		if (_read_reg((ixpci_canreg_t *) ioctl_param, dp))
		//if (_read_reg((ixpci_reg_t *) ioctl_param, dp->ioaddr, (dp->canportid+1), dp->caninfo->baud))
			return FAILURE;
		break;
	case IXPCI_WRITE_REG:
		if (_write_reg((ixpci_canreg_t *) ioctl_param, dp->ioaddr, (dp->canportid+1)))
			return FAILURE;
		break;
	case IXPCI_IRQ_ENABLE:
                _enable_irq(dp);
                break;
        case IXPCI_IRQ_DISABLE:
                _disable_irq(dp);
                break;
	default:
		return -EINVAL;
	}
	return SUCCESS;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,0)
int ixpisocan200_release(struct inode *inode, struct file *file)
#else
void ixpisocan200_release(struct inode *inode, struct file *file)
#endif

{
	/* (export)
	 *
	 * This function is called by ixpci.o whenever a process attempts to
	 * closes the device file. It doesn't have a return value in kernel
	 * version 2.0.x because it can't fail (you must always be able to
	 * close a device).  In version 2.2.x it is allowed to fail.
	 *
	 * Arguments: read <linux/fs.h> for (*release) of struct file_operations
	 *
	 * Returned:  none */

	int minor, bar;
	ixpci_devinfo_t *dp;

	minor = MINOR(inode->i_rdev);
	dp = _align_minor(minor);
	
	if (dp)
	{
		if(file->private_data)
		{
			file->private_data = NULL;
                	kfree(file->private_data);
                }

		dp->open = 0;

		bar = dp->canportid+1;

		/* change to reset mode to stop the operation of chip */
		iowrite8( (ioread8(dp->ioaddr[bar] + MOD) | CAN_RESET_REQUEST), dp->ioaddr[bar] + MOD );

		iowrite8( 0, dp->ioaddr[bar] + IER );

		release_mem_region(dp->base[bar], dp->range[bar]);
                iounmap(dp->ioaddr[bar]);
//		dp->ioaddr[bar] = NULL;

		/* uninstall ISR */
	        free_irq(dp->irq, dp);
# if LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,0) && LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
                MOD_DEC_USE_COUNT;
#endif

	}


# if LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,0)
	return 0;
# endif
}

static struct file_operations fops = {
        /* kernel 2.6 prevent the module from unloading while there is a open file(kernel 2.4 use the funciton MOD_INC_USE_COUNT and MOD_DEC_USE_COUNT to protect module from unloading when someone is opening file), so driver writer must set the file_operation's field owner a value 'THIS_MODULE' */
# if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
        owner:THIS_MODULE,
#endif
	open:ixpisocan200_open,
        release:ixpisocan200_release,

# if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,36)
        unlocked_ioctl:ixpisocan200_ioctl,
#else
        ioctl:ixpisocan200_ioctl,
#endif
        read:ixpisocan200_read,
        write:ixpisocan200_write,
};

int ixpisocan200_open(struct inode *inode, struct file *file)
{
	/* (export)
	 *
	 * This function is called by ixpci.o whenever a process attempts to
	 * open the device file of PCI-1602
	 *
	 * Arguments: read <linux/fs.h> for (*open) of struct file_operations
	 *
	 * Returned:  none */

	int minor, i, bar, port;
	ixpci_devinfo_t *dp;
	unsigned int timeout = 0;
	unsigned int result = SUCCESS;
	unsigned int card_no;

	minor = MINOR(inode->i_rdev);
	dp = _align_minor(minor);

	if (!dp) return -EINVAL;

	//add struct dp for system call read and write function
	file->private_data = dp;

	++(dp->open);
	if (dp->open > 1) {
		--(dp->open);
		return -EBUSY;
		/* if still opened by someone, get out */
	}

	card_no = dp->cno - 1;
	bar = dp->canportid + 1;
	port = dp->canportid;

	//printk("ixpisocan200_open port : %d bar : %d dp->no : %d\n", port, bar, dp->no);

	/* request io region */
        //Bar 0 (PLX), Bar 2 (port 1),  Bar 3 ~ 5   (port 2 ~ 4)
 
	if (!isInitIOMem[card_no])
        {
	        if(request_mem_region(dp->base[0], dp->range[0], MODULE_NAME) == NULL)
		{
			//printk("request_mem_region bar %d fail \n",i);
			viraddr[card_no] = NULL;
		}
		else
		{
			//printk("request_mem_region bar %d OK \n",i);
       			viraddr[card_no] = ioremap(dp->base[0], dp->range[0]);
		}

		isInitIOMem[card_no] = 1;
	}

	if(request_mem_region(dp->base[bar], dp->range[bar], MODULE_NAME) == NULL)
        {
		return FAILURE;
        }
        else
        {
                //printk("request_mem_region bar %d OK \n",i);
                dp->ioaddr[bar] = ioremap(dp->base[bar], dp->range[bar]);
        }

/*
//	iowrite8(ResetReg[port], viraddr[2] + 0x1ff);

//	while(++timeout < CAN_TIMEOUT_CNT)
//	{	// wait Hardware Reset
//		if( ioread8(dp->ioaddr[bar] + MOD) & CAN_RESET_REQUEST ) break;		
//      }
//
*/
	/* tx fifo needs only to be inizialized the first time. CAN is opened */

	dp->caninfo->tx_buf.head   = 0;
       	dp->caninfo->tx_buf.tail   = 0;
       	dp->caninfo->tx_buf.status = 0;
       	dp->caninfo->tx_buf.active = 0;
       	
	for(i = 0; i < MAX_BUFSIZE; i++)
	{
       		dp->caninfo->tx_buf.free[i]  = BUF_EMPTY;
        }

	init_waitqueue_head(&CanOutWait[card_no][port-1]);
	spin_lock_init(&write_splock[card_no][port-1]);
	
        /* initialize and mark used rx buffer in flags field */

	dp->caninfo->rx_buf.head   = 0;
        dp->caninfo->rx_buf.tail   = 0;
        dp->caninfo->rx_buf.status = 0;
        dp->caninfo->rx_buf.active = 0;
	init_waitqueue_head(&CanWait[card_no][port-1]);
	CanWaitFlag[card_no][port-1] = 1;

	
	/* init CAN sja1000 chip */
	// MOD : set 'reset' mode

        iowrite8( (CAN_RESET_REQUEST | CAN_ACC_FILT_MASK), dp->ioaddr[bar] + MOD);

        timeout = 0;
        while(++timeout < CAN_TIMEOUT_CNT)
        {
                if( ioread8(dp->ioaddr[bar] + MOD) & CAN_RESET_REQUEST ) break;
        }
	if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

	// CDR : choose BasicCAN or PeliCAN mode 

        iowrite8((CAN_MODE_PELICAN + CAN_MODE_CLK), dp->ioaddr[bar] + CDR );

	timeout = 0;
	while(++timeout < CAN_TIMEOUT_CNT)
        {       
                if( ioread8(dp->ioaddr[bar] + CDR) == (CAN_MODE_PELICAN + CAN_MODE_CLK) ) break;
        }
	if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

	// IER : disable CAN interrupt 
	iowrite8(0x00, dp->ioaddr[bar] + IER );

	timeout = 0;
        while(++timeout < CAN_TIMEOUT_CNT)
        {
                if( ioread8(dp->ioaddr[bar] + IER) == 0x00 ) break;
        }
	if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

	// RBSA : init RBSA 0 
	iowrite8(0, dp->ioaddr[bar] + RBSA );

        timeout = 0;
        while(++timeout < CAN_TIMEOUT_CNT)
        {
                if( ioread8(dp->ioaddr[bar] + RBSA) == 0x00 ) break;
        }
	if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

	/* start CAN sja1000 chip */

	//set Acceptance Code

        iowrite32(dp->caninfo->acr, dp->ioaddr[bar] + PACR);
	timeout = 0;
        while(++timeout < CAN_TIMEOUT_CNT)
        {
                if( ioread32(dp->ioaddr[bar] + PACR) == dp->caninfo->acr  ) break;
        }
	if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;
	//printk("ACR value : 0x%x  <--> ACR register : 0x%x\n", dp->caninfo->acr,ioread32(dp->ioaddr[bar] + PACR));

	//set Acceptance Mask

        iowrite32(dp->caninfo->amr, dp->ioaddr[bar] + PAMR);
	timeout = 0;
        while(++timeout < CAN_TIMEOUT_CNT)
        {
                if( ioread32(dp->ioaddr[bar] + PAMR) == dp->caninfo->amr  ) break;
        }
	if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;
	//printk("AMR value : 0x%x  <--> AMR register : 0x%x\n", dp->caninfo->amr,ioread32(dp->ioaddr[bar] + PAMR));

	// set CAN sja1000 Bus Timing 0 and Bus Timing 1

        switch(dp->caninfo->baud)
        {
                case   10: i = 0; break;
                case   20: i = 1; break;
                case   50: i = 2; break;
                case  100: i = 3; break;
                case  125: i = 4; break;
                case  250: i = 5; break;        //default baud
                case  500: i = 6; break;
                case  800: i = 7; break;
                case 1000: i = 8; break;
                default  : i = 5; break;
        }

        iowrite8(CanTiming[i][0], dp->ioaddr[bar] + BTR0);
	timeout = 0;
        while(++timeout < CAN_TIMEOUT_CNT)
        {
                if( ioread8(dp->ioaddr[bar] + BTR0 ) == CanTiming[i][0]  ) break;
        }
	if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

        iowrite8(CanTiming[i][1], dp->ioaddr[bar] + BTR1);
	timeout = 0;
        while(++timeout < CAN_TIMEOUT_CNT)
        {
                if( ioread8(dp->ioaddr[bar] + BTR1 ) == CanTiming[i][1] ) break;
        }
	if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

	// set OCR
	iowrite8(CAN_OUTC_VAL, dp->ioaddr[bar] + OCR);
	timeout = 0;
        while(++timeout < CAN_TIMEOUT_CNT)
        {
                if( ioread8(dp->ioaddr[bar] + OCR ) == CAN_OUTC_VAL ) break;
        }
	if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

	// clear interrupts 
	ioread8(dp->ioaddr[bar] + IR);

	//Interrupt on TX and data overrun
        iowrite8( (ioread8(dp->ioaddr[bar] + IER) | CAN_TRANSMIT_INT_ENABLE | CAN_OVERRUN_INT_ENABLE), dp->ioaddr[bar] + IER);
        timeout = 0;
        while(++timeout < CAN_TIMEOUT_CNT)
        {
                if(ioread8(dp->ioaddr[bar] + IER) == (CAN_TRANSMIT_INT_ENABLE|CAN_OVERRUN_INT_ENABLE))  break;
        }
        if ( timeout >= CAN_TIMEOUT_CNT ) result = FAILURE;

        // MOD : change to operating mode	
	iowrite8( (ioread8(dp->ioaddr[bar] + MOD) & ~CAN_RESET_REQUEST), dp->ioaddr[bar] + MOD );

	/* Install ISR */

        if (request_irq(dp->irq, irq_handler, SA_SHIRQ, dp->name, dp)) 
	{
                --(dp->open);
                return -EBUSY;
        }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
        MOD_INC_USE_COUNT;
#endif
	return result;
}

void cleanup_module()
{
	/* cleanup this module */

	ixpci_devinfo_t *dp;

	KMSG("%s ", MODULE_NAME);

	for (dp = dev; dp; dp = dp->next_f)
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
        	cdev_del(dp->cdev);
#else
	        dp->fops = 0;
#endif

		if (isInitIOMem[dp->cno-1])
		{
//			for (i = 0; i < PBAN; i++)
//        		{
//        			if ( i == 1 ) continue;
		        release_mem_region(dp->base[0], dp->range[0]);
			iounmap(viraddr[dp->cno-1]);

//			}

			isInitIOMem[dp->cno-1] = 0;
	  	}

		
		/* remove file operations */
		printk(".");
	}
	printk(" has been removed.\n");
}

int init_module()
{
	/* initialize this module
	 *
	 * Arguments: none
	 *
	 * Returned:
	 * integer 0 for ok, otherwise failed (module can't be load) */

	ixpci_devinfo_t *dp;

	KMSG("%s ", MODULE_NAME);

	/* align to first PISO-CAN200 in ixpci list */
	for (dev = ixpci_dev; dev && (dev->id != PISO_CAN200) && (dev->id != PISO_CAN200U) && (dev->id != PISO_CAN200E) && (dev->id != PCM_CAN200); dev = dev->next);

	if (!dev) {
		printk("fail!\n");
		return FAILURE;
	}

	/* initiate for each device (card) in family */
	for (dp = dev; dp; dp = dp->next_f) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
                dp->cdev->ops = &fops;
#else
                dp->fops = &fops;
#endif
		//_reset_dev(dp);
		printk(".");
	}

	printk(" ready.\n");
	return SUCCESS;
}
